///////////////
// View List //
///////////////

#include "Card List.h"
#include "Card View.h"
#include "Filter.h"
#include "Library.h"
#include "Dialogs.h"
#include "Basic IO.h"
#include <stdlib.h>
#include <strstrea.h>

const int SAVED_END = 0;

extern BOOL Nice;

static const int SPLASHPAUSE = 4;

ViewList Views;

ViewList::ViewList ()
{
    _Loaded = FALSE;
    _Help = FALSE;
}

void ViewList::Load (fstream& file)
{
    char name [256];
    ViewItem *view;
    int index;

    for (index = 0; index < List.GetCount (); index ++) {
        view = (ViewItem *) List.GetItemData (index);
        view->LoadData (file);
    }

    _Loaded = TRUE;
}

void ViewList::Save ()
{
    char name [256];
    fstream file;
    ViewItem *view;
    int index;

    LoadString (IDS_DATAFILE, name);
    file.open (name, ios::in | ios::out | ios::trunc);

    if (Nice)
        file << "NICE" << endl;
    file << SIGNATURE << endl;

    WriteRect (file, _Rect);
    file << endl;

    for (index = 0; index < List.GetCount (); index ++) {
        view = (ViewItem *) List.GetItemData (index);
        view->Save (file);
    }

    file << SAVED_END << endl;

    for (index = 0; index < List.GetCount (); index ++) {
        view = (ViewItem *) List.GetItemData (index);
        view->SaveData (file);
    }
}

void ViewList::Add (ViewItem *view)
{
    List.AddItemData ((LPARAM) view);
}

void ViewList::Remove (ViewItem *view)
{
    int index;

    index = 0;

    while (List.GetItemData (index) != (LPARAM) view)
        index ++;

    List.DeleteString (index);
}

void ViewList::Modify (int key)
{
    ViewItem *view;
    int index;

    for (index = 0; index < List.GetCount (); index ++) {
        view = (ViewItem *) List.GetItemData (index);
        view->ModifyCard (key);
    }
}

void ViewList::Delete (int key)
{
    ViewItem *view;
    int index;

    for (index = 0; index < List.GetCount (); index ++) {
        view = (ViewItem *) List.GetItemData (index);
        view->DeleteCard (key);
    }
}

void ViewList::Refresh (ViewItem *view)
{
    Remove (view);
    Add (view);
}

ViewItem * ViewList::Find (POINT& point)
{
    ViewItem *view;
    HWND window;
    int index;

    window = WindowFromPoint (point);

    index = 0;

    while (TRUE)
        if (index == List.GetCount ())
            return NULL;
        else {
            view = (ViewItem *) List.GetItemData (index);
            if ((*view == window) || view->IsChild (window))
                return view;
            else
                index ++;
        }
}

void ViewList::Enable (BOOL enable)
{
    ViewItem *view;
    int index;

    index = List.GetCount () - 1;

    if (index >= 0)
        do {
            view = (ViewItem *) List.GetItemData (index);
            view->EnableWindow (enable);
        } while (index --);

    EnableWindow (enable);
}

BOOL ViewList::LoadWindows (fstream& file)
{
    char nice [256];
    char name [256];
    RECT rect;
    ViewItem *view;
    int signature;
    int type;

    if (file) {
        file.getline (nice, sizeof (nice));
        if (!stricmp (nice, "NICE")) {
            Nice = TRUE;
            file >> signature;
        } else
            signature = atoi (nice);
        if (signature != SIGNATURE)
            return FALSE;
        ReadRect (file, rect);
    } else {
        rect.left = 0;
        rect.top = 0;
        rect.right = GetSystemMetrics (SM_CXSCREEN) / 2;
        rect.bottom = GetSystemMetrics (SM_CYSCREEN) / 2;
        OffsetRect (&rect, rect.right / 2, rect.bottom / 2);
    }
    MoveWindow (rect);
    AdjustWindow (*this);

    BOOL done = FALSE;

    while (!done && (file >> type))
        switch (type) {
            case SAVED_CARD:
                view = AllocCardView ();
                view->Load (file);
                break;
            case SAVED_DECK:
                view = AllocCardList ();
                view->Load (file);
                break;
            case SAVED_END:
                done = TRUE;
                break;
        }
    return TRUE;
}

static struct LOADPARAM
{
    char Name [256];
    CardList *NewList;
    CardList *UpdateList;
};

static DWORD pascal LoadCards (void *params)
{
    epp_ListBox newlist;
    epp_ListBox updatelist;
    LOADPARAM *load = (LOADPARAM *) params;

    newlist.Attach (load->NewList->GetCardList ());
    updatelist.Attach (load->UpdateList->GetCardList ());

    Library.Import (newlist, updatelist, load->Name);
    load->NewList->UpdateStats ();
    load->UpdateList->UpdateStats ();
    BusyEnd ();

    delete load;
    return 0;
}

void ViewList::ImportCards ()
{
    ImportDialog dialog;
    epp_Thread thread;
    CardList *newdeck, *updatedeck;
    LOADPARAM *load;

    if (dialog.Run (*this)) {
        newdeck = AllocCardList ();
        newdeck->Create ("New Cards", TRUE, TRUE);
        newdeck->Open ();
        updatedeck = AllocCardList ();
        updatedeck->Create ("Updated Cards", TRUE, TRUE);
        updatedeck->Open ();

        load = new LOADPARAM;
        strcpy (load->Name, dialog);
        load->NewList = newdeck;
        load->UpdateList = updatedeck;
        thread.Create (LoadCards, load);
        BusyBegin (IDS_LOADINGCARDS, *this);
    }
}

void ViewList::NewDeck ()
{
    char title [256];
    CardList *deck;

    LoadString (IDS_UNTITLED, title);
    deck = AllocCardList ();
    deck->Create (title, FALSE, FALSE);
    deck->Open ();
}

int ViewList::NewCard (HWND parent, BOOL open)
{
    CardDialog dialog;
    CardView *view;
    CARD_INFO card;
    int key;

    memset (&card, 0, sizeof (card));
    if (dialog.Run (&card, parent) && strlen (card.Name)) {
        key = Library.New (card);
        if (open) {
            view = AllocCardView ();
            if (view) {
                view->Create (key);
                view->Open ();
            }
        }
        return key;
    } else
        return 0;
}

static struct LOADDECK
{
    CardList *List;
    CardList *Errors;
    char *Name;
    BOOL Direct;
};

static DWORD pascal LoadDeck (void *params)
{
    epp_ListBox load;
    epp_ListBox errors;
    ListFilter filter;
    CARD_INFO card;
    char name [256];
    int count;
    int ecount;
    int key;

    LOADDECK *deck = (LOADDECK *) params;

    load.Attach (deck->List->GetCardList ());
    errors.Attach (deck->Errors->GetCardList ());

    fstream file;
    strstream data;
    istream *stream;

    if (deck->Direct) {
        data.write (deck->Name, strlen (deck->Name));
        stream = &data;
    } else {
        file.open (deck->Name, ios::in);
        stream = &file;
    }
    stream->seekg (0);

    filter.ReadHeader (*stream, name);
    deck->List->SetTitle (name);
    Views.Refresh (deck->List);

    ecount = 0;

    while (filter.Read (*stream, count, name)) {
        key = Library.Find (name);
        if (!key) {
            memset (&card, 0, sizeof (card));
            strcpy (card.Name, name);
            key = Library.New (card);
            errors.AddItemData (key);
            ecount ++;
        }
        load.AddItemData (key | ((count - 1) << 16));
    }
    deck->List->UpdateStats ();

    if (ecount) {
        deck->Errors->Open ();
        deck->List->UpdateStats ();
    } else
        deck->Errors->PostMessage (ITEM_CLOSE);

    delete deck->Name;
    delete deck;
    BusyEnd ();
    return 0;
}

void ViewList::ImportDeck (HWND parent, BOOL clipboard)
{
    epp_Thread thread;
    LOADDECK *deck;

    if (clipboard) {
        HANDLE block;
        char *src, *tar;
        void *memptr;
        unsigned size;

        if (!Check ("Begin deck import from clipboard?", parent))
            return;
        if (!OpenClipboard (*this))
            return;
        block = GetClipboardData (CF_TEXT);
        if (!block) {
            CloseClipboard ();
            return;
        }

        deck = new LOADDECK;
        memptr = GlobalLock (block);
        size = strlen ((char *) memptr);
        deck->Name = new char [size + 1];
        strcpy (deck->Name, (char *) memptr);
        GlobalUnlock (block);
        CloseClipboard ();
        src = tar = deck->Name;
        while (*src) {
            if (*src != '\r')
                *tar ++ = *src;
            src ++;
        }
        *tar = NULL;
        deck->Direct = TRUE;
    } else {
        ImportDeckDialog dialog;
        char message [256];

        if (!dialog.Run (parent))
            return;
        wsprintf (message, "Begin deck import from \"%s\"?", (const char *) dialog);
        if (!Check (message, parent))
            return;

        deck = new LOADDECK;
        deck->Name = new char [256];
        strcpy (deck->Name, dialog);
        deck->Direct = FALSE;
    }

    deck->List = AllocCardList ();
    deck->List->Create ("Imported Deck", FALSE, TRUE);
    deck->List->Open ();
    deck->Errors = AllocCardList ();
    deck->Errors->Create ("Unrecognized Cards", FALSE, TRUE);
    thread.Create (LoadDeck, deck);
    BusyBegin (IDS_IMPORTDECK, *deck->List);
}

void ViewList::AllCards ()
{
    char title [256];
    CardList *deck;

    deck = AllocCardList ();
    if (deck) {
        LoadString (IDS_ALLCARDS, title);
        deck->Create (title, TRUE, TRUE);
        deck->Open ();
        SearchLibrary (NULL, deck, NULL);
    }
}

void ViewList::FindCards ()
{
    SearchDialog dialog;
    CardList *deck;

    if (dialog.Run (*this, "Search Library")) {
        deck = AllocCardList ();
        if (deck) {
            deck->Create (SearchTitle (dialog), TRUE, TRUE);
            deck->Open ();
            SearchLibrary (NULL, deck, dialog);
        }
    }
}

void ViewList::CloseAll ()
{
    ViewItem *view;
    int index;

    index = List.GetCount () - 1;

    if (index >= 0)
        do {
            view = (ViewItem *) List.GetItemData (index);
            view->Hide ();
        } while (index --);
}

void ViewList::Help (BOOL open)
{
    char file [256];
    LoadString (IDS_HELPFILE, file);
    WinHelp (file, (open)? HELP_FINDER : HELP_QUIT);
    _Help = TRUE;
}

HMENU ViewList::GetContextMenu ()
{
    ViewItem *view = GetCurView ();

    if (view)
        return view->GetMenu ();
    else
        return NULL;
}

ViewItem * ViewList::GetCurView ()
{
    ViewItem *view;
    int index;

    view = NULL;

    if (List.GetCount ()) {
        index = List.GetCurSel ();
        if (index != LB_ERR)
            view = (ViewItem *) List.GetItemData (index);
    }

    return view;
}

void ViewList::SetRect ()
{
    if (!IsIconic () && !IsZoomed ())
        GetWindowRect (&_Rect);
}

void ViewList::OnCommand (int notify, int id, HWND)
{
    ViewItem *view = GetCurView ();

    switch (id) {
        case IDM_MAIN_IMPORT:
            ImportCards ();
            break;
        case IDM_MAIN_LOADDECK:
            ImportDeck (*this, FALSE);
            break;
        case IDM_MAIN_LOADDECKCLIP:
            ImportDeck (*this, TRUE);
            break;
        case IDM_MAIN_NEWDECK:
            NewDeck ();
            break;
        case IDM_MAIN_NEWCARD:
            NewCard (*this);
            break;
        case IDM_MAIN_OPEN:
        case IDM_MAIN_OPENCARD:
        case IDM_MAIN_OPENDECK:
            if (view) {
                SetWindowPos (HWND_BOTTOM, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
                view->Open ();
            }
            break;
        case IDM_MAIN_CLOSECARD:
        case IDM_MAIN_CLOSEDECK:
        case IDM_MAIN_CLOSE:
            if (view)
                view->Hide ();
            break;
        case IDM_MAIN_CLOSEALL:
            CloseAll ();
            break;
        case IDM_MAIN_RENAMEDECK:
            if (view)
                view->Rename (*this);
            break;
        case IDM_MAIN_DELETEDECK:
        case IDM_MAIN_DELETE:
            if (view)
                view->Delete (*this);
            break;
        case IDM_MAIN_SEARCH:
            FindCards ();
            break;
        case IDM_MAIN_SHOWALL:
            AllCards ();
            break;
        case IDM_MAIN_EXIT:
            Close ();
            break;
        case IDM_MAIN_HELP:
            Help (TRUE);
            break;
        case IDM_MAIN_ABOUT:
            AboutDialog dialog;
            dialog.Run (*this);
            break;
        case IDD_LIST:
            if (notify == LBN_DBLCLK)
                if (view) {
                    SetWindowPos (HWND_BOTTOM, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
                    view->Open ();
                }
            break;
    }
}

int ViewList::OnCompareItem (int, COMPAREITEMSTRUCT *data)
{
    ViewItem *view1, *view2;
    char title1 [256], title2 [256];

    view1 = (ViewItem *) data->itemData1;
    view2 = (ViewItem *) data->itemData2;

    title1 [0] = view1->Sort ();
    view1->GetWindowText (title1 + 1, sizeof (title1) - 1);
    title2 [0] = view2->Sort ();
    view2->GetWindowText (title2 + 1, sizeof (title2) - 1);
    return strcmpi (title1, title2);
}

static DWORD pascal LoadData (void *param)
{
    fstream *file = (fstream *) param;
    char message [256];

    LoadString (IDS_LOADCARD, message);
    Splash.Message (message);
    Library.Load ();
    LoadString (IDS_LOADDECK, message);
    Splash.Message (message);
    Views.Load (*file);
    Splash.Message (NULL);
    delete file;
    return 0;
}

int ViewList::OnCreate (CREATESTRUCT *)
{
    epp_Thread thread;
    char name [256];
    char mess [256];
    RECT trect;
    RECT rect;

    Splash.Create ();
    _TimerCount = 0;
    _Timer.Create (1000, *this);

    LoadString (IDS_MAINFONT, name);
    Font.Create (BASE, name);

	Menu.LoadMenu (IDR_MAIN);
	SetMenu (Menu);

	const int COMMANDS [7] = { NULL, IDM_MAIN_NEWDECK, NULL, IDM_MAIN_OPEN, IDM_MAIN_SEARCH, NULL, IDM_MAIN_DELETE };
	Toolbar.Create (*this, CCS_NOMOVEY | TBSTYLE_TOOLTIPS, IDB_MAINBAR, COMMANDS, 7);
	Toolbar.ShowWindow ();
	Toolbar.SetDelayTime (2500);
	GetClientRect (&trect);

    GetClientRect (&rect);
    rect.top += trect.bottom + 1;
    rect.bottom -= trect.bottom;
    List.Create (LBS_NOINTEGRALHEIGHT | LBS_OWNERDRAWFIXED | LBS_SORT | LBS_NOTIFY | WS_DLGFRAME | WS_VSCROLL | WS_VISIBLE, rect, *this, IDD_LIST);
    sList.Create (List, this, NULL);
    Width = rect.right;

    LoadString (IDS_LOADWIN, name);
    Splash.Message (name);

    fstream *file = new fstream;
    LoadString (IDS_DATAFILE, name);
    file->open (name, ios::in);

    if (LoadWindows (*file)) {
        _Save = TRUE;
        thread.Create (LoadData, file);
        SetRect ();
    } else {
        _Save = FALSE;
        Splash.Close ();
        wsprintf (mess, "Error in file:  %s", name);
        Error (mess, MB_ICONSTOP);
        Close ();
    }

    return 0;
}

void ViewList::OnDestroy ()
{
    ViewItem *view;

    CloseAll ();

    if (_Save) {
        Views.Save ();
        Library.Save ();
    }

    while (List.GetCount ()) {
        view = (ViewItem *) List.GetItemData (0);
        view->Delete (NULL);
    }

    Font.Close ();
    Toolbar.Close ();
    List.Close ();

    if (_Help)
        Help (FALSE);
}

void ViewList::OnDrawItem (int, DRAWITEMSTRUCT *data)
{
    epp_Context dc;
    char title [256];
    HFONT oldfont;
    ViewItem *view;
    RECT rect;

    dc.Attach (data->hDC);

    if ((data->itemAction == ODA_DRAWENTIRE) || (data->itemAction == ODA_SELECT))
        if (List.GetCount ()) {
            rect = data->rcItem;
            rect.right ++;
            rect.bottom ++;
            dc.FillRect (&rect, dc.GetCurrentObject (OBJ_BRUSH));
            rect.right --;
            rect.bottom --;
            view = (ViewItem *) data->itemData;
            view->GetWindowText (title, sizeof (title));
            rect.left += BASE / 2;
            oldfont = dc.SelectObject (Font);
            dc.DrawText (title, -1, &rect, DT_LEFT | DT_VCENTER | DT_NOPREFIX);
            dc.SelectObject (oldfont);
            if (data->itemState & ODS_SELECTED)
                dc.DrawFocusRect (&data->rcItem);
        }
}

BOOL ViewList::OnEraseBkgnd (HDC)
{
    return TRUE;
}

void ViewList::OnMeasureItem (int, MEASUREITEMSTRUCT *data)
{
    data->itemWidth = Width;
    data->itemHeight = BASE + EXTRA;
}

void ViewList::OnMove (int, int)
{
    SetRect ();
}

void ViewList::OnSetFocus (HWND)
{
    SetFocus (List);
}

void ViewList::OnSize (int, int, int)
{
    RECT trect;
    RECT rect;

    Toolbar.SendMessage (WM_SIZE, _wParam, _lParam);

    GetClientRect (&rect);
    Toolbar.GetClientRect (&trect);
    rect.top += trect.bottom + 1;
    List.MoveWindow (rect);
    Width = rect.right;
}

void ViewList::OnTimer (int)
{
    _TimerCount ++;

    if (Splash)
        if ((_TimerCount > SPLASHPAUSE) && Library.Loaded () && Views.Loaded ()) {
            _Timer.Close ();
            Splash.Close ();
            ShowWindow ();
        }
}

BEGIN_HANDLER (ViewList, epp_Frame)
    ON_WM_DRAWITEM
    ON_WM_COMPAREITEM
    ON_WM_MEASUREITEM
    ON_WM_COMMAND
    ON_WM_ERASEBKGND
    ON_WM_SETFOCUS
    ON_WM_MOVE
    ON_WM_SIZE
    ON_WM_TIMER
    ON_WM_CREATE
    ON_WM_DESTROY
END_HANDLER